<?php

namespace App\Controllers;

use App\Models\BiodataModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\HTTP\Files\UploadedFile;

class BiodataAdmin extends BaseController
{
    private BiodataModel $biodata;

    public function __construct()
    {
        $this->biodata = new BiodataModel();
    }

    public function index(): string
    {
        $search         = trim((string) $this->request->getGet('q'));
        $headlineFilter = trim((string) $this->request->getGet('headline'));
        $sort           = (string) $this->request->getGet('sort');

        $query = $this->biodata;

        if ($search !== '') {
            $query->groupStart()
                ->like('full_name', $search)
                ->orLike('headline', $search)
                ->orLike('skills', $search)
                ->orLike('summary', $search)
                ->groupEnd();
        }

        if ($headlineFilter !== '' && $headlineFilter !== 'semua') {
            $query->where('headline', $headlineFilter);
        }

        $sort = $sort !== '' ? $sort : 'newest';
        switch ($sort) {
            case 'name_asc':
                $query->orderBy('full_name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('full_name', 'desc');
                break;
            default:
                $query->orderBy('created_at', 'desc');
                $sort = 'newest';
        }

        $profiles = $query->paginate(10, 'biodata');

        $headlineOptions = (new BiodataModel())
            ->select('headline')
            ->distinct()
            ->orderBy('headline', 'asc')
            ->findAll();
        $headlineOptions = array_values(array_filter(array_map(static function ($row) {
            return $row['headline'] ?? null;
        }, $headlineOptions)));

        return view('admin_list_biodata', [
            'profiles'        => $profiles,
            'pager'           => $this->biodata->pager,
            'headlineOptions' => $headlineOptions,
            'filters'         => [
                'search'   => $search,
                'headline' => $headlineFilter !== '' ? $headlineFilter : 'semua',
                'sort'     => $sort,
            ],
        ]);
    }

    public function create()
    {
        if ($this->request->is('post')) {
            $rules = $this->rules();

            if (! $this->validate($rules)) {
                return view('admin_form_biodata', [
                    'mode'       => 'create',
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $photoFile = $this->request->getFile('photo');
            $photoPath = $this->storePhoto($photoFile);

            $payload = [
                'full_name' => (string) $this->request->getPost('full_name'),
                'headline'  => (string) $this->request->getPost('headline'),
                'summary'   => (string) $this->request->getPost('summary') ?: null,
                'address'   => (string) $this->request->getPost('address') ?: null,
                'phone'     => (string) $this->request->getPost('phone') ?: null,
                'email'     => (string) $this->request->getPost('email') ?: null,
                'skills'    => (string) $this->request->getPost('skills') ?: null,
                'photo_path'=> $photoPath,
            ];

            $this->biodata->insert($payload);

            return redirect()
                ->to('/admin/biodata')
                ->with('message', 'Biodata berhasil ditambahkan.');
        }

        return view('admin_form_biodata', [
            'mode' => 'create',
        ]);
    }

    public function edit(int $id)
    {
        $profile = $this->findOrFail($id);

        if ($this->request->is('post')) {
            $rules = $this->rules();

            if (! $this->validate($rules)) {
                return view('admin_form_biodata', [
                    'mode'       => 'edit',
                    'profile'    => $profile,
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $photoFile   = $this->request->getFile('photo');
            $newPhoto    = $this->storePhoto($photoFile);
            $removePhoto = (bool) $this->request->getPost('remove_photo');

            $payload = [
                'full_name' => (string) $this->request->getPost('full_name'),
                'headline'  => (string) $this->request->getPost('headline'),
                'summary'   => (string) $this->request->getPost('summary') ?: null,
                'address'   => (string) $this->request->getPost('address') ?: null,
                'phone'     => (string) $this->request->getPost('phone') ?: null,
                'email'     => (string) $this->request->getPost('email') ?: null,
                'skills'    => (string) $this->request->getPost('skills') ?: null,
            ];

            if ($newPhoto !== null) {
                if (! empty($profile['photo_path'])) {
                    $this->deletePhoto($profile['photo_path']);
                }
                $payload['photo_path'] = $newPhoto;
            } elseif ($removePhoto) {
                $this->deletePhoto($profile['photo_path'] ?? null);
                $payload['photo_path'] = null;
            }

            $this->biodata->update($id, $payload);

            return redirect()
                ->to('/admin/biodata')
                ->with('message', 'Biodata berhasil diperbarui.');
        }

        return view('admin_form_biodata', [
            'mode'    => 'edit',
            'profile' => $profile,
        ]);
    }

    public function delete(int $id)
    {
        $profile = $this->findOrFail($id);
        $this->biodata->delete($id);

        if (! empty($profile['photo_path'])) {
            $this->deletePhoto($profile['photo_path']);
        }

        return redirect()
            ->to('/admin/biodata')
            ->with('message', 'Biodata berhasil dihapus.');
    }

    private function rules(): array
    {
        return [
            'full_name' => [
                'rules'  => 'required|string',
                'errors' => [
                    'required' => 'Nama wajib diisi.',
                ],
            ],
            'headline' => [
                'rules'  => 'required|string',
                'errors' => [
                    'required' => 'Headline wajib diisi.',
                ],
            ],
            'email' => [
                'rules'  => 'permit_empty|valid_email',
                'errors' => [
                    'valid_email' => 'Format email tidak valid.',
                ],
            ],
            'skills' => [
                'rules' => 'permit_empty|string',
            ],
            'photo' => [
                'rules'  => 'permit_empty|is_image[photo]|max_size[photo,4096]|mime_in[photo,image/jpg,image/jpeg,image/png,image/webp]',
                'errors' => [
                    'is_image' => 'Foto harus berupa gambar.',
                    'max_size' => 'Ukuran foto maksimal 4 MB.',
                    'mime_in'  => 'Format foto harus JPG, JPEG, PNG, atau WEBP.',
                ],
            ],
        ];
    }

    private function storePhoto(?UploadedFile $photoFile): ?string
    {
        if (! $photoFile instanceof UploadedFile) {
            return null;
        }

        if ($photoFile->getError() === UPLOAD_ERR_NO_FILE) {
            return null;
        }

        if (! $photoFile->isValid() || $photoFile->hasMoved()) {
            log_message('error', 'Upload foto biodata tidak valid: {error}', [
                'error' => $photoFile->getErrorString(),
            ]);

            return null;
        }

        $targetDir = FCPATH . 'uploads/biodata';
        if (! is_dir($targetDir) && ! mkdir($targetDir, 0755, true) && ! is_dir($targetDir)) {
            log_message('error', 'Gagal membuat direktori foto biodata: {dir}', ['dir' => $targetDir]);
            return null;
        }

        $newName = $photoFile->getRandomName();

        try {
            $photoFile->move($targetDir, $newName);
        } catch (\Throwable $th) {
            log_message('error', 'Gagal memindahkan foto biodata: {message}', ['message' => $th->getMessage()]);
            return null;
        }

        return 'uploads/biodata/' . $newName;
    }

    private function deletePhoto(?string $relativePath): void
    {
        if (! $relativePath) {
            return;
        }

        $fullPath = FCPATH . ltrim(str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $relativePath), DIRECTORY_SEPARATOR);
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    private function findOrFail(int $id): array
    {
        $profile = $this->biodata->find($id);
        if (! $profile) {
            throw PageNotFoundException::forPageNotFound();
        }

        return $profile;
    }
}
