<?php

namespace App\Controllers;

use App\Models\NewsModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\HTTP\Files\UploadedFile;

class NewsAdmin extends BaseController
{
    private NewsModel $news;
    private bool $hasImageColumn = false;

    public function __construct()
    {
        helper(['text']);
        $this->news = new NewsModel();
        $db = db_connect();
        $this->hasImageColumn = $db->fieldExists('image_path', $this->news->getTable());
    }

    public function index(): string
    {
        $newses = $this->news->orderBy('created_at', 'desc')->findAll();

        return view('admin_list_news', [
            'newses' => $newses,
        ]);
    }

    public function preview(int $id): string
    {
        $news = $this->findOrFail($id);

        return view('news_detail', ['news' => $news]);
    }

    public function create()
    {
        if ($this->request->is('post')) {
            $postData   = $this->request->getPost();
            $postStatus = (string) ($postData['action'] ?? $postData['status'] ?? 'draft');
            if (! in_array($postStatus, ['published', 'draft'], true)) {
                $postStatus = 'draft';
            }
            $validatedInput = array_merge($postData, ['status' => $postStatus]);

            $rules = [
                'title'   => 'required|string',
                'content' => 'required',
                'status'  => 'required|in_list[published,draft]',
            ];

            if ($this->hasImageColumn) {
                $rules['image'] = 'permit_empty|is_image[image]|max_size[image,2048]|mime_in[image,image/jpg,image/jpeg,image/png,image/webp]';
            }

            if (! $this->validate($rules, [], $validatedInput)) {
                return view('admin_create_news', [
                    'validation' => $this->validator,
                    'oldInput'   => $validatedInput,
                    'imageColumnAvailable' => $this->hasImageColumn,
                ]);
            }

            $title     = (string) ($validatedInput['title'] ?? '');
            $status    = $postStatus;
            $payload = [
                'title'      => $title,
                'content'    => (string) ($validatedInput['content'] ?? ''),
                'status'     => $status,
                'slug'       => url_title($title, '-', true),
            ];

            if ($this->hasImageColumn) {
                $imageFile = $this->request->getFile('image');
                $imagePath = $this->storeNewsImage($imageFile);
                $payload['image_path'] = $imagePath;
            }

            $this->news->insert($payload);
            cache()->clean(); // pastikan tidak ada data lama yang tersimpan di cache

            return redirect()
                ->to('/admin/news')
                ->with('message', 'Berita berhasil dibuat.');
        }

        return view('admin_create_news', [
            'imageColumnAvailable' => $this->hasImageColumn,
        ]);
    }

    public function edit(int $id)
    {
        $news = $this->findOrFail($id);

        if ($this->request->is('post')) {
            $postData   = $this->request->getPost();
            $postStatus = (string) ($postData['action'] ?? $postData['status'] ?? '');
            if (! in_array($postStatus, ['published', 'draft'], true)) {
                $postStatus = $news['status'];
            }
            $validatedInput = array_merge($postData, ['status' => $postStatus]);

            $rules = [
                'title'   => 'required|string',
                'content' => 'required',
                'status'  => 'required|in_list[published,draft]',
            ];

            if ($this->hasImageColumn) {
                $rules['image'] = 'permit_empty|is_image[image]|max_size[image,2048]|mime_in[image,image/jpg,image/jpeg,image/png,image/webp]';
            }

            if (! $this->validate($rules, [], $validatedInput)) {
                return view('admin_edit_news', [
                    'news'       => $news,
                    'validation' => $this->validator,
                    'oldInput'   => array_merge($validatedInput, [
                        'remove_image' => $this->request->getPost('remove_image'),
                    ]),
                    'imageColumnAvailable' => $this->hasImageColumn,
                ]);
            }

            $title        = (string) ($validatedInput['title'] ?? '');
            $status       = $postStatus;
            $payload      = [
                'title'   => $title,
                'content' => (string) ($validatedInput['content'] ?? ''),
                'status'  => $status,
                'slug'    => url_title($title, '-', true),
            ];

            if ($this->hasImageColumn) {
                $removeImage  = (bool) $this->request->getPost('remove_image');
                $imageFile    = $this->request->getFile('image');
                $currentImage = $news['image_path'] ?? null;

                $newImagePath = $this->storeNewsImage($imageFile);
                if ($newImagePath !== null) {
                    if (! empty($currentImage)) {
                        $this->deleteNewsImage($currentImage);
                    }
                    $payload['image_path'] = $newImagePath;
                } elseif ($removeImage && ! empty($currentImage)) {
                    $this->deleteNewsImage($currentImage);
                    $payload['image_path'] = null;
                }
            }

            if (! $this->news->update($id, $payload)) {
                log_message('error', 'Gagal update berita {id}: {errors}', [
                    'id'     => $id,
                    'errors' => json_encode($this->news->errors()),
                ]);

                return redirect()
                    ->back()
                    ->withInput()
                    ->with('errors', ['update' => 'Gagal menyimpan perubahan. Coba lagi.']);
            }

            cache()->clean(); // bersihkan cache supaya status terbaru terlihat

            return redirect()
                ->to("/admin/news/{$id}/edit")
                ->with('message', 'Berita berhasil diperbarui. Status: ' . ucfirst($status) . '.');
        }

        return view('admin_edit_news', [
            'news' => $news,
            'imageColumnAvailable' => $this->hasImageColumn,
        ]);
    }

    public function delete(int $id)
    {
        $this->news->delete($id);
        cache()->clean(); // pastikan daftar berita tidak terbaca dari cache

        return redirect()
            ->to('/admin/news')
            ->with('message', 'Berita dihapus.');
    }

    public function publish(int $id)
    {
        return $this->updateStatus($id, 'published', 'Berita diterbitkan.');
    }

    public function draft(int $id)
    {
        return $this->updateStatus($id, 'draft', 'Berita disimpan sebagai draft.');
    }

    private function storeNewsImage(?UploadedFile $imageFile): ?string
    {
        if (! $imageFile instanceof UploadedFile) {
            return null;
        }

        if ($imageFile->getError() === UPLOAD_ERR_NO_FILE) {
            return null;
        }

        if (! $imageFile->isValid() || $imageFile->hasMoved()) {
            log_message('error', 'Upload gambar berita tidak valid: {error}', [
                'error' => $imageFile->getErrorString(),
            ]);

            return null;
        }

        $targetDir = FCPATH . 'uploads/news';
        if (! is_dir($targetDir) && ! mkdir($targetDir, 0755, true) && ! is_dir($targetDir)) {
            log_message('error', 'Gagal membuat direktori penyimpanan gambar berita: {dir}', ['dir' => $targetDir]);

            return null;
        }

        $newName = $imageFile->getRandomName();

        try {
            $imageFile->move($targetDir, $newName);
        } catch (\Throwable $th) {
            log_message('error', 'Gagal memindahkan gambar berita: {message}', ['message' => $th->getMessage()]);

            return null;
        }

        return 'uploads/news/' . $newName;
    }

    private function deleteNewsImage(?string $relativePath): void
    {
        if (! $relativePath) {
            return;
        }

        $fullPath = FCPATH . ltrim(str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $relativePath), DIRECTORY_SEPARATOR);
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    private function updateStatus(int $id, string $status, ?string $message = null)
    {
        $news = $this->findOrFail($id);
        $payload = [
            'status' => $status,
            'slug'   => url_title($news['title'], '-', true),
        ];

        if (! $this->news->update($id, $payload)) {
            log_message('error', 'Gagal ubah status berita {id}: {errors}', [
                'id'     => $id,
                'errors' => json_encode($this->news->errors()),
            ]);

            return redirect()
                ->back()
                ->with('errors', ['update' => 'Gagal mengubah status. Coba lagi.']);
        }

        cache()->clean();

        $flashMessage = $message ?? ('Status berita diperbarui menjadi ' . $status . '.');

        return redirect()
            ->to('/admin/news')
            ->with('message', $flashMessage);
    }

    private function findOrFail(int $id): array
    {
        $news = $this->news->where('id', $id)->first();

        if (! $news) {
            throw PageNotFoundException::forPageNotFound();
        }

        return $news;
    }
}
