<?php 

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\DailyActivityModel;
use CodeIgniter\Exceptions\PageNotFoundException;

class ActivitiesAdmin extends BaseController
{
    protected $activitiesModel;
    protected $session;

    public function __construct()
    {
        $this->activitiesModel = new DailyActivityModel();
        $this->session = \Config\Services::session();
    }

    public function index()
    {
        // Get search parameter
        $search = $this->request->getGet('search');
        
        // Get filter parameters
        $filter_tanggal = $this->request->getGet('filter_tanggal');
        
        // Get sorting parameters
        $sort_by = $this->request->getGet('sort_by') ?? 'tanggal';
        $sort_order = $this->request->getGet('sort_order') ?? 'DESC';
        
        // Apply search
        if (!empty($search)) {
            $this->activitiesModel->groupStart()
                ->like('nama_aktivitas', $search)
                ->orLike('deskripsi', $search)
                ->groupEnd();
        }
        
        // Apply filter
        if (!empty($filter_tanggal)) {
            $this->activitiesModel->where('tanggal', $filter_tanggal);
        }
        
        // Apply sorting
        $this->activitiesModel->orderBy($sort_by, $sort_order);
        
        $data = [
            'title' => 'Kelola Aktivitas',
            'activities' => $this->activitiesModel->paginate(10),
            'pager' => $this->activitiesModel->pager,
            'search' => $search,
            'filter_tanggal' => $filter_tanggal,
            'sort_by' => $sort_by,
            'sort_order' => $sort_order,
        ];
        
        return view('admin/activities/index', $data);
    }

    public function preview($id)
    {
        $activity = $this->activitiesModel->where('id', $id)->first();
        
        if (!$activity) {
            throw PageNotFoundException::forPageNotFound();
        }

        $data = [
            'title' => 'Preview Aktivitas',
            'activity' => $activity
        ];

        return view('admin/activities/preview', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Aktivitas',
            'validation' => \Config\Services::validation()
        ];

        // lakukan validasi
        $validation = \Config\Services::validation();
        $validation->setRules([
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'tanggal' => 'required|valid_date',
            'jam' => 'required'
        ]);
        
        $isDataValid = $validation->withRequest($this->request)->run();

        // jika data valid, simpan ke database
        if ($isDataValid) {
            // Handle file upload for foto
            $foto = '';
            $fileFoto = $this->request->getFile('foto');
            if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
                $newName = $fileFoto->getRandomName();
                $fileFoto->move(WRITEPATH . '../public/uploads/', $newName);
                $foto = 'uploads/' . $newName;
            }
            
            // Handle file upload for video
            $video = '';
            $fileVideo = $this->request->getFile('video');
            if ($fileVideo && $fileVideo->isValid() && !$fileVideo->hasMoved()) {
                $newName = $fileVideo->getRandomName();
                $fileVideo->move(WRITEPATH . '../public/uploads/', $newName);
                $video = 'uploads/' . $newName;
            }
            
            $this->activitiesModel->insert([
                "nama_aktivitas" => $this->request->getPost('nama_aktivitas'),
                "tanggal" => $this->request->getPost('tanggal'),
                "jam" => $this->request->getPost('jam'),
                "deskripsi" => $this->request->getPost('deskripsi'),
                "foto" => $foto,
                "video" => $video
            ]);
            
            return redirect()->to('/admin/activities')->with('success', 'Aktivitas berhasil ditambahkan.');
        }
        
        return view('admin/activities/create', $data);
    }

    public function edit($id)
    {
        // ambil data yang akan diedit
        $activity = $this->activitiesModel->where('id', $id)->first();
        
        if (!$activity) {
            return redirect()->to('/admin/activities')->with('error', 'Aktivitas tidak ditemukan.');
        }

        $data = [
            'title' => 'Edit Aktivitas',
            'activity' => $activity,
            'validation' => \Config\Services::validation()
        ];
        
        // Check if this is a POST request (form submission)
        if ($this->request->getMethod() === 'POST') {
            // lakukan validasi data
            $validation = \Config\Services::validation();
            $validation->setRules([
                'id' => 'required',
                'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
                'tanggal' => 'required|valid_date',
                'jam' => 'required'
            ]);
            
            $isDataValid = $validation->withRequest($this->request)->run();
            
            // jika data valid, maka simpan ke database
            if ($isDataValid) {
                $updateData = [
                    "nama_aktivitas" => $this->request->getPost('nama_aktivitas'),
                    "tanggal" => $this->request->getPost('tanggal'),
                    "jam" => $this->request->getPost('jam'),
                    "deskripsi" => $this->request->getPost('deskripsi')
                ];
                
                // Handle file upload for foto
                $fileFoto = $this->request->getFile('foto');
                if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
                    // Delete old foto if exists
                    if (!empty($activity['foto'])) {
                        $oldFile = WRITEPATH . '../public/' . $activity['foto'];
                        if (file_exists($oldFile)) {
                            unlink($oldFile);
                        }
                    }
                    $newName = $fileFoto->getRandomName();
                    $fileFoto->move(WRITEPATH . '../public/uploads/', $newName);
                    $updateData['foto'] = 'uploads/' . $newName;
                }
                
                // Handle file upload for video
                $fileVideo = $this->request->getFile('video');
                if ($fileVideo && $fileVideo->isValid() && !$fileVideo->hasMoved()) {
                    // Delete old video if exists
                    if (!empty($activity['video'])) {
                        $oldFile = WRITEPATH . '../public/' . $activity['video'];
                        if (file_exists($oldFile)) {
                            unlink($oldFile);
                        }
                    }
                    $newName = $fileVideo->getRandomName();
                    $fileVideo->move(WRITEPATH . '../public/uploads/', $newName);
                    $updateData['video'] = 'uploads/' . $newName;
                }
                
                $this->activitiesModel->update($id, $updateData);
                return redirect()->to('/admin/activities')->with('success', 'Aktivitas berhasil diupdate.');
            }
            
            // If validation failed, set validation errors in data
            $data['validation'] = $validation;
        }

        return view('admin/activities/edit', $data);
    }

    public function delete($id)
    {
        $activity = $this->activitiesModel->where('id', $id)->first();
        
        if (!$activity) {
            return redirect()->to('/admin/activities')->with('error', 'Aktivitas tidak ditemukan.');
        }
        
        // Delete foto if exist
        if (!empty($activity['foto'])) {
            $file = WRITEPATH . '../public/' . $activity['foto'];
            if (file_exists($file)) {
                unlink($file);
            }
        }
        
        // Delete video if exist
        if (!empty($activity['video'])) {
            $file = WRITEPATH . '../public/' . $activity['video'];
            if (file_exists($file)) {
                unlink($file);
            }
        }
        
        $this->activitiesModel->delete($id);
        return redirect()->to('/admin/activities')->with('success', 'Aktivitas berhasil dihapus.');
    }
}