<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\AdminModel;

class AdminManagement extends BaseController
{
    protected $adminModel;
    protected $session;

    public function __construct()
    {
        $this->adminModel = new AdminModel();
        $this->session = \Config\Services::session();
        
        // Load admin helper (sudah auto-load via Autoload.php)
        // helper('admin');
    }

    /**
     * List all admins (Super Admin only)
     */
    public function index()
    {
        // Check if user is super admin
        if (!is_super_admin()) {
            return redirect()->to('/admin/dashboard')->with('error', 'Akses ditolak. Hanya Super Admin yang dapat mengelola admin.');
        }

        $data = [
            'title' => 'Kelola Admin',
            'admins' => $this->adminModel->findAll(),
            'current_admin' => get_admin_data()
        ];

        return view('admin/management/index', $data);
    }

    /**
     * Create new admin form
     */
    public function create()
    {
        // Check permission
        if (!admin_can_access('admin_create')) {
            return redirect()->to('/admin/dashboard')->with('error', 'Anda tidak memiliki akses untuk membuat admin baru.');
        }

        $data = [
            'title' => 'Tambah Admin Baru',
            'validation' => \Config\Services::validation()
        ];

        return view('admin/management/create', $data);
    }

    /**
     * Store new admin
     */
    public function store()
    {
        // Check permission
        if (!admin_can_access('admin_create')) {
            return redirect()->to('/admin/dashboard')->with('error', 'Akses ditolak.');
        }

        $rules = [
            'username' => 'required|min_length[3]|max_length[100]|is_unique[admins.username]',
            'email' => 'required|valid_email|is_unique[admins.email]',
            'password' => 'required|min_length[6]',
            'confirm_password' => 'required|matches[password]',
            'full_name' => 'required|min_length[3]|max_length[255]',
            'role' => 'required|in_list[super_admin,admin]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $data = [
            'username' => $this->request->getPost('username'),
            'email' => $this->request->getPost('email'),
            'password' => $this->request->getPost('password'),
            'full_name' => $this->request->getPost('full_name'),
            'role' => $this->request->getPost('role'),
            'is_active' => 1
        ];

        if ($this->adminModel->createAdmin($data)) {
            return redirect()->to('/admin/management')->with('success', 'Admin baru berhasil ditambahkan.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan admin baru.');
        }
    }

    /**
     * Edit admin form
     */
    public function edit($id)
    {
        // Check permission
        if (!admin_can_access('admin_edit')) {
            return redirect()->to('/admin/dashboard')->with('error', 'Anda tidak memiliki akses untuk mengedit admin.');
        }

        $admin = $this->adminModel->find($id);
        if (!$admin) {
            return redirect()->to('/admin/management')->with('error', 'Admin tidak ditemukan.');
        }

        // Prevent editing super admin by regular admin
        if ($admin['role'] === 'super_admin' && !is_super_admin()) {
            return redirect()->to('/admin/management')->with('error', 'Anda tidak dapat mengedit Super Admin.');
        }

        $data = [
            'title' => 'Edit Admin',
            'admin' => $admin,
            'validation' => \Config\Services::validation()
        ];

        return view('admin/management/edit', $data);
    }

    /**
     * Update admin
     */
    public function update($id)
    {
        // Check permission
        if (!admin_can_access('admin_edit')) {
            return redirect()->to('/admin/dashboard')->with('error', 'Akses ditolak.');
        }

        $admin = $this->adminModel->find($id);
        if (!$admin) {
            return redirect()->to('/admin/management')->with('error', 'Admin tidak ditemukan.');
        }

        // Prevent editing super admin by regular admin
        if ($admin['role'] === 'super_admin' && !is_super_admin()) {
            return redirect()->to('/admin/management')->with('error', 'Anda tidak dapat mengedit Super Admin.');
        }

        $rules = [
            'username' => "required|min_length[3]|max_length[100]|is_unique[admins.username,id,{$id}]",
            'email' => "required|valid_email|is_unique[admins.email,id,{$id}]",
            'full_name' => 'required|min_length[3]|max_length[255]',
            'role' => 'required|in_list[super_admin,admin]'
        ];

        // Add password validation if password is provided
        if ($this->request->getPost('password')) {
            $rules['password'] = 'min_length[6]';
            $rules['confirm_password'] = 'matches[password]';
        }

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $data = [
            'username' => $this->request->getPost('username'),
            'email' => $this->request->getPost('email'),
            'full_name' => $this->request->getPost('full_name'),
            'role' => $this->request->getPost('role'),
            'is_active' => $this->request->getPost('is_active') ? 1 : 0
        ];

        // Add password if provided
        if ($this->request->getPost('password')) {
            $data['password'] = $this->request->getPost('password');
        }

        if ($this->adminModel->updateAdmin($id, $data)) {
            return redirect()->to('/admin/management')->with('success', 'Data admin berhasil diupdate.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate data admin.');
        }
    }

    /**
     * Delete admin
     */
    public function delete($id)
    {
        // Check permission
        if (!admin_can_access('admin_delete')) {
            return redirect()->to('/admin/dashboard')->with('error', 'Anda tidak memiliki akses untuk menghapus admin.');
        }

        $admin = $this->adminModel->find($id);
        if (!$admin) {
            return redirect()->to('/admin/management')->with('error', 'Admin tidak ditemukan.');
        }

        // Prevent deleting super admin
        if ($admin['role'] === 'super_admin') {
            return redirect()->to('/admin/management')->with('error', 'Super Admin tidak dapat dihapus.');
        }

        // Prevent self deletion
        if ($admin['id'] == get_admin_data('id')) {
            return redirect()->to('/admin/management')->with('error', 'Anda tidak dapat menghapus akun sendiri.');
        }

        if ($this->adminModel->delete($id)) {
            return redirect()->to('/admin/management')->with('success', 'Admin berhasil dihapus.');
        } else {
            return redirect()->to('/admin/management')->with('error', 'Gagal menghapus admin.');
        }
    }

    /**
     * Toggle admin status
     */
    public function toggleStatus($id)
    {
        // Check permission
        if (!is_super_admin()) {
            return redirect()->to('/admin/management')->with('error', 'Hanya Super Admin yang dapat mengubah status admin.');
        }

        $admin = $this->adminModel->find($id);
        if (!$admin) {
            return redirect()->to('/admin/management')->with('error', 'Admin tidak ditemukan.');
        }

        // Prevent deactivating super admin
        if ($admin['role'] === 'super_admin') {
            return redirect()->to('/admin/management')->with('error', 'Status Super Admin tidak dapat diubah.');
        }

        // Prevent self deactivation
        if ($admin['id'] == get_admin_data('id')) {
            return redirect()->to('/admin/management')->with('error', 'Anda tidak dapat menonaktifkan akun sendiri.');
        }

        $newStatus = $admin['is_active'] ? 0 : 1;
        
        if ($this->adminModel->update($id, ['is_active' => $newStatus])) {
            $statusText = $newStatus ? 'diaktifkan' : 'dinonaktifkan';
            return redirect()->to('/admin/management')->with('success', "Admin berhasil {$statusText}.");
        } else {
            return redirect()->to('/admin/management')->with('error', 'Gagal mengubah status admin.');
        }
    }
}