<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\AdminModel;

class Auth extends BaseController
{
    protected $adminModel;
    protected $session;

    public function __construct()
    {
        $this->adminModel = new AdminModel();
        $this->session = \Config\Services::session();
    }

    /**
     * Display login form
     */
    public function login()
    {
        // Redirect if already logged in
        if ($this->session->get('admin_logged_in')) {
            return redirect()->to('/admin/dashboard');
        }

        $data = [
            'title' => 'Admin Login',
            'validation' => \Config\Services::validation()
        ];

        return view('admin/auth/login', $data);
    }

    /**
     * Process login
     */
    public function processLogin()
    {
        $rules = [
            'username' => 'required',
            'password' => 'required'
        ];

        $messages = [
            'username' => [
                'required' => 'Username atau email harus diisi'
            ],
            'password' => [
                'required' => 'Password harus diisi'
            ]
        ];

        if (!$this->validate($rules, $messages)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $username = $this->request->getPost('username');
        $password = $this->request->getPost('password');
        $remember = $this->request->getPost('remember');

        $admin = $this->adminModel->authenticate($username, $password);

        if ($admin) {
            // Set session data
            $sessionData = [
                'admin_id' => $admin['id'],
                'admin_username' => $admin['username'],
                'admin_email' => $admin['email'],
                'admin_full_name' => $admin['full_name'],
                'admin_role' => $admin['role'],
                'admin_logged_in' => true
            ];

            $this->session->set($sessionData);

            // Set remember me cookie if checked
            if ($remember) {
                $cookieValue = base64_encode($admin['id'] . ':' . $admin['username']);
                setcookie('admin_remember', $cookieValue, time() + (30 * 24 * 60 * 60), '/'); // 30 days
            }

            return redirect()->to('/admin/dashboard')->with('success', 'Login berhasil! Selamat datang ' . $admin['full_name']);
        } else {
            return redirect()->back()->withInput()->with('error', 'Username/email atau password salah');
        }
    }

    /**
     * Logout admin
     */
    public function logout()
    {
        // Remove session data
        $this->session->remove([
            'admin_id',
            'admin_username', 
            'admin_email',
            'admin_full_name',
            'admin_role',
            'admin_logged_in'
        ]);

        // Remove remember me cookie
        setcookie('admin_remember', '', time() - 3600, '/');

        return redirect()->to('/admin/login')->with('success', 'Anda telah logout');
    }

    /**
     * Check remember me cookie
     */
    public function checkRememberMe()
    {
        if (isset($_COOKIE['admin_remember']) && !$this->session->get('admin_logged_in')) {
            $cookieValue = base64_decode($_COOKIE['admin_remember']);
            $parts = explode(':', $cookieValue);
            
            if (count($parts) === 2) {
                $adminId = $parts[0];
                $username = $parts[1];
                
                $admin = $this->adminModel->getAdmin($adminId);
                
                if ($admin && $admin['username'] === $username) {
                    // Auto login
                    $sessionData = [
                        'admin_id' => $admin['id'],
                        'admin_username' => $admin['username'],
                        'admin_email' => $admin['email'],
                        'admin_full_name' => $admin['full_name'],
                        'admin_role' => $admin['role'],
                        'admin_logged_in' => true
                    ];

                    $this->session->set($sessionData);
                    return redirect()->to('/admin/dashboard');
                }
            }
        }
        
        return redirect()->to('/admin/login');
    }
}