<?php 

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\BiodataModel;
use CodeIgniter\Exceptions\PageNotFoundException;

class BiodataAdmin extends BaseController
{
    protected $biodataModel;
    protected $session;

    public function __construct()
    {
        $this->biodataModel = new BiodataModel();
        $this->session = \Config\Services::session();
    }

    public function index()
    {
        // Get search parameter
        $search = $this->request->getGet('search');
        
        // Get filter parameters
        $filter_jenis_kelamin = $this->request->getGet('filter_jenis_kelamin');
        
        // Get sorting parameters
        $sort_by = $this->request->getGet('sort_by') ?? 'nama';
        $sort_order = $this->request->getGet('sort_order') ?? 'ASC';
        
        // Apply search
        if (!empty($search)) {
            $this->biodataModel->groupStart()
                ->like('nama', $search)
                ->orLike('alamat', $search)
                ->orLike('email', $search)
                ->groupEnd();
        }
        
        // Apply filter
        if (!empty($filter_jenis_kelamin)) {
            $this->biodataModel->where('jenis_kelamin', $filter_jenis_kelamin);
        }
        
        // Apply sorting
        $this->biodataModel->orderBy($sort_by, $sort_order);
        
        $data = [
            'title' => 'Kelola Biodata',
            'biodata' => $this->biodataModel->paginate(10),
            'pager' => $this->biodataModel->pager,
            'search' => $search,
            'filter_jenis_kelamin' => $filter_jenis_kelamin,
            'sort_by' => $sort_by,
            'sort_order' => $sort_order,
        ];
        
        return view('admin/biodata/index', $data);
    }

    public function preview($id)
    {
        $biodata = $this->biodataModel->where('id', $id)->first();
        
        if (!$biodata) {
            throw PageNotFoundException::forPageNotFound();
        }

        $data = [
            'title' => 'Preview Biodata',
            'biodata' => $biodata
        ];

        return view('admin/biodata/preview', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Biodata',
            'validation' => \Config\Services::validation()
        ];

        // lakukan validasi
        $validation = \Config\Services::validation();
        $validation->setRules([
            'nama' => 'required|min_length[3]|max_length[255]',
            'email' => 'required|valid_email',
            'no_hp' => 'required',
            'jenis_kelamin' => 'required|in_list[Laki-laki,Perempuan]'
        ]);
        
        $isDataValid = $validation->withRequest($this->request)->run();

        // jika data valid, simpan ke database
        if ($isDataValid) {
            // Handle file upload for foto_profil
            $foto_profil = '';
            $fileFoto = $this->request->getFile('foto_profil');
            if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
                $newName = $fileFoto->getRandomName();
                $fileFoto->move(WRITEPATH . '../public/uploads/', $newName);
                $foto_profil = 'uploads/' . $newName;
            }
            
            $this->biodataModel->insert([
                "nama" => $this->request->getPost('nama'),
                "alamat" => $this->request->getPost('alamat'),
                "tempat_lahir" => $this->request->getPost('tempat_lahir'),
                "tanggal_lahir" => $this->request->getPost('tanggal_lahir'),
                "jenis_kelamin" => $this->request->getPost('jenis_kelamin'),
                "agama" => $this->request->getPost('agama'),
                "no_hp" => $this->request->getPost('no_hp'),
                "email" => $this->request->getPost('email'),
                "foto_profil" => $foto_profil,
                "deskripsi" => $this->request->getPost('deskripsi')
            ]);
            
            return redirect()->to('/admin/biodata')->with('success', 'Biodata berhasil ditambahkan.');
        }
        
        return view('admin/biodata/create', $data);
    }

    public function edit($id)
    {
        // ambil data yang akan diedit
        $biodata = $this->biodataModel->where('id', $id)->first();
        
        if (!$biodata) {
            return redirect()->to('/admin/biodata')->with('error', 'Biodata tidak ditemukan.');
        }

        $data = [
            'title' => 'Edit Biodata',
            'biodata' => $biodata,
            'validation' => \Config\Services::validation()
        ];
        
        // Check if this is a POST request (form submission)
        if ($this->request->getMethod() === 'POST') {
            // lakukan validasi data
            $validation = \Config\Services::validation();
            $validation->setRules([
                'id' => 'required',
                'nama' => 'required|min_length[3]|max_length[255]',
                'email' => 'required|valid_email',
                'no_hp' => 'required',
                'jenis_kelamin' => 'required|in_list[Laki-laki,Perempuan]'
            ]);
            
            $isDataValid = $validation->withRequest($this->request)->run();
            
            // jika data valid, maka simpan ke database
            if ($isDataValid) {
                $updateData = [
                    "nama" => $this->request->getPost('nama'),
                    "alamat" => $this->request->getPost('alamat'),
                    "tempat_lahir" => $this->request->getPost('tempat_lahir'),
                    "tanggal_lahir" => $this->request->getPost('tanggal_lahir'),
                    "jenis_kelamin" => $this->request->getPost('jenis_kelamin'),
                    "agama" => $this->request->getPost('agama'),
                    "no_hp" => $this->request->getPost('no_hp'),
                    "email" => $this->request->getPost('email'),
                    "deskripsi" => $this->request->getPost('deskripsi')
                ];
                
                // Handle file upload for foto_profil
                $fileFoto = $this->request->getFile('foto_profil');
                if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
                    // Delete old foto if exists
                    if (!empty($biodata['foto_profil'])) {
                        $oldFile = WRITEPATH . '../public/' . $biodata['foto_profil'];
                        if (file_exists($oldFile)) {
                            unlink($oldFile);
                        }
                    }
                    $newName = $fileFoto->getRandomName();
                    $fileFoto->move(WRITEPATH . '../public/uploads/', $newName);
                    $updateData['foto_profil'] = 'uploads/' . $newName;
                }
                
                $this->biodataModel->update($id, $updateData);
                return redirect()->to('/admin/biodata')->with('success', 'Biodata berhasil diupdate.');
            }
            
            // If validation failed, set validation errors in data
            $data['validation'] = $validation;
        }

        return view('admin/biodata/edit', $data);
    }

    public function delete($id)
    {
        $biodata = $this->biodataModel->where('id', $id)->first();
        
        if (!$biodata) {
            return redirect()->to('/admin/biodata')->with('error', 'Biodata tidak ditemukan.');
        }
        
        // Delete file if exist
        if (!empty($biodata['foto_profil'])) {
            $file = WRITEPATH . '../public/' . $biodata['foto_profil'];
            if (file_exists($file)) {
                unlink($file);
            }
        }
        
        $this->biodataModel->delete($id);
        return redirect()->to('/admin/biodata')->with('success', 'Biodata berhasil dihapus.');
    }
}

