<?php

namespace App\Models;

use CodeIgniter\Model;

class AdminModel extends Model
{
    protected $table            = 'admins';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'username',
        'email', 
        'password',
        'full_name',
        'role',
        'is_active',
        'last_login'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';

    // Validation
    protected $validationRules = [
        'username' => 'required|min_length[3]|max_length[100]|is_unique[admins.username,id,{id}]',
        'email'    => 'required|valid_email|is_unique[admins.email,id,{id}]',
        'password' => 'required|min_length[6]',
        'full_name'=> 'required|min_length[3]|max_length[255]',
        'role'     => 'required|in_list[super_admin,admin]'
    ];

    protected $validationMessages = [
        'username' => [
            'required'    => 'Username harus diisi',
            'min_length'  => 'Username minimal 3 karakter',
            'max_length'  => 'Username maksimal 100 karakter',
            'is_unique'   => 'Username sudah digunakan'
        ],
        'email' => [
            'required'    => 'Email harus diisi',
            'valid_email' => 'Format email tidak valid',
            'is_unique'   => 'Email sudah digunakan'
        ],
        'password' => [
            'required'    => 'Password harus diisi',
            'min_length'  => 'Password minimal 6 karakter'
        ],
        'full_name' => [
            'required'    => 'Nama lengkap harus diisi',
            'min_length'  => 'Nama lengkap minimal 3 karakter',
            'max_length'  => 'Nama lengkap maksimal 255 karakter'
        ]
    ];

    protected $skipValidation = false;

    /**
     * Authenticate admin login
     */
    public function authenticate($username, $password)
    {
        $admin = $this->where('username', $username)
                     ->orWhere('email', $username)
                     ->where('is_active', 1)
                     ->first();

        if ($admin && password_verify($password, $admin['password'])) {
            // Update last login
            $this->update($admin['id'], ['last_login' => date('Y-m-d H:i:s')]);
            return $admin;
        }

        return false;
    }

    /**
     * Get admin by ID
     */
    public function getAdmin($id)
    {
        return $this->where('id', $id)->where('is_active', 1)->first();
    }

    /**
     * Create new admin with hashed password
     */
    public function createAdmin($data)
    {
        if (isset($data['password'])) {
            $data['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
        }
        
        return $this->insert($data);
    }

    /**
     * Update admin with password hashing if needed
     */
    public function updateAdmin($id, $data)
    {
        if (isset($data['password']) && !empty($data['password'])) {
            $data['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
        } else {
            unset($data['password']);
        }
        
        return $this->update($id, $data);
    }
}