<?php

namespace App\Models;

use CodeIgniter\Model;

class EducationModel extends Model
{
    protected $table = 'education';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'institusi',
        'jurusan',
        'gelar',
        'tahun_mulai',
        'tahun_selesai',
        'deskripsi'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    // Validation
    protected $validationRules = [
        'institusi' => 'required|min_length[3]|max_length[255]',
        'jurusan' => 'required|max_length[255]',
        'gelar' => 'required|max_length[255]',
        'tahun_mulai' => 'required|integer|greater_than[1900]',
        'tahun_selesai' => 'permit_empty|integer|greater_than[1900]'
    ];

    protected $validationMessages = [
        'institusi' => [
            'required' => 'Nama institusi harus diisi',
            'min_length' => 'Nama institusi minimal 3 karakter',
            'max_length' => 'Nama institusi maksimal 255 karakter'
        ],
        'jurusan' => [
            'required' => 'Jurusan harus diisi',
            'max_length' => 'Jurusan maksimal 255 karakter'
        ],
        'gelar' => [
            'required' => 'Gelar harus diisi',
            'max_length' => 'Gelar maksimal 255 karakter'
        ],
        'tahun_mulai' => [
            'required' => 'Tahun mulai harus diisi',
            'integer' => 'Tahun mulai harus berupa angka',
            'greater_than' => 'Tahun mulai tidak valid'
        ],
        'tahun_selesai' => [
            'integer' => 'Tahun selesai harus berupa angka',
            'greater_than' => 'Tahun selesai tidak valid'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Get education with search functionality
     */
    public function getEducationWithSearch($search = null, $limit = 10, $offset = 0)
    {
        $builder = $this->builder();
        
        if ($search) {
            $builder->groupStart()
                ->like('institusi', $search)
                ->orLike('jurusan', $search)
                ->orLike('gelar', $search)
                ->groupEnd();
        }
        
        return $builder->orderBy('tahun_mulai', 'DESC')
                      ->limit($limit, $offset)
                      ->get()
                      ->getResultArray();
    }

    /**
     * Get education by degree level
     */
    public function getEducationByDegree($degree)
    {
        return $this->like('gelar', $degree)
                   ->orderBy('tahun_mulai', 'DESC')
                   ->findAll();
    }

    /**
     * Get current education (ongoing)
     */
    public function getCurrentEducation()
    {
        return $this->where('tahun_selesai IS NULL OR tahun_selesai = ""')
                   ->orderBy('tahun_mulai', 'DESC')
                   ->findAll();
    }

    /**
     * Get completed education
     */
    public function getCompletedEducation()
    {
        return $this->where('tahun_selesai IS NOT NULL AND tahun_selesai != ""')
                   ->orderBy('tahun_selesai', 'DESC')
                   ->findAll();
    }
}