<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ActivityModel;

class ActivityAdminController extends BaseController
{
    protected $activityModel;

    public function __construct()
    {
        $this->activityModel = new ActivityModel();
        helper('media');
    }

    public function index()
    {
        $activities = $this->activityModel->orderBy('created_at', 'DESC')->findAll();

        $data = [
            'title'      => 'Kelola Aktivitas',
            'activities' => $activities,
        ];

        return view('admin/activity/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Aktivitas',
        ];

        return view('admin/activity/create', $data);
    }

    public function store()
    {
        $rules = [
            'title'    => 'required|max_length[255]',
            'category' => 'permit_empty|max_length[100]',
            'content'  => 'permit_empty',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'title'         => $this->request->getPost('title'),
            'slug'          => url_title($this->request->getPost('title'), '-', true),
            'activity_date' => $this->request->getPost('activity_date') ?: date('Y-m-d'),
            'category'      => $this->request->getPost('category'),
            'content'       => $this->request->getPost('content'),
            'status'        => $this->request->getPost('status') ?: 'draft',
        ];

        // Handle tags
        $tags = $this->request->getPost('tags');
        if ($tags) {
            $data['tags_json'] = json_encode(array_filter(array_map('trim', explode(',', $tags))));
        }

        // Handle thumbnail
        $thumbnail = $this->request->getFile('thumbnail');
        if ($thumbnail && $thumbnail->isValid() && !$thumbnail->hasMoved()) {
            $data['thumbnail'] = upload_media($thumbnail, 'activities');
        }

        // Ensure unique slug
        $baseSlug = $data['slug'];
        $counter = 1;
        while ($this->activityModel->where('slug', $data['slug'])->first()) {
            $data['slug'] = $baseSlug . '-' . $counter++;
        }

        if ($this->activityModel->insert($data)) {
            return redirect()->to('/admin/activities')->with('success', 'Aktivitas berhasil ditambahkan!');
        }

        return redirect()->back()->withInput()->with('error', 'Gagal menambahkan aktivitas.');
    }

    public function edit($id)
    {
        $activity = $this->activityModel->find($id);

        if (!$activity) {
            return redirect()->to('/admin/activities')->with('error', 'Aktivitas tidak ditemukan.');
        }

        // Decode tags for form
        $activity['tags'] = json_decode($activity['tags_json'] ?? '[]', true);

        $data = [
            'title'    => 'Edit Aktivitas',
            'activity' => $activity,
        ];

        return view('admin/activity/edit', $data);
    }

    public function update($id)
    {
        $activity = $this->activityModel->find($id);

        if (!$activity) {
            return redirect()->to('/admin/activities')->with('error', 'Aktivitas tidak ditemukan.');
        }

        $rules = [
            'title'    => 'required|max_length[255]',
            'category' => 'permit_empty|max_length[100]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'title'         => $this->request->getPost('title'),
            'activity_date' => $this->request->getPost('activity_date') ?: date('Y-m-d'),
            'category'      => $this->request->getPost('category'),
            'content'       => $this->request->getPost('content'),
            'status'        => $this->request->getPost('status') ?: 'draft',
        ];

        // Handle slug if title changed
        if ($this->request->getPost('title') !== $activity['title']) {
            $newSlug = url_title($this->request->getPost('title'), '-', true);
            $baseSlug = $newSlug;
            $counter = 1;
            while ($this->activityModel->where('slug', $newSlug)->where('id !=', $id)->first()) {
                $newSlug = $baseSlug . '-' . $counter++;
            }
            $data['slug'] = $newSlug;
        }

        // Handle tags
        $tags = $this->request->getPost('tags');
        $data['tags_json'] = json_encode(array_filter(array_map('trim', explode(',', $tags ?? ''))));

        // Handle thumbnail
        $thumbnail = $this->request->getFile('thumbnail');
        if ($thumbnail && $thumbnail->isValid() && !$thumbnail->hasMoved()) {
            // Delete old thumbnail
            if (!empty($activity['thumbnail'])) {
                delete_media($activity['thumbnail']);
            }
            $data['thumbnail'] = upload_media($thumbnail, 'activities');
        }

        if ($this->activityModel->update($id, $data)) {
            return redirect()->to('/admin/activities')->with('success', 'Aktivitas berhasil diperbarui!');
        }

        return redirect()->back()->withInput()->with('error', 'Gagal memperbarui aktivitas.');
    }

    public function delete($id)
    {
        $activity = $this->activityModel->find($id);

        if (!$activity) {
            return redirect()->to('/admin/activities')->with('error', 'Aktivitas tidak ditemukan.');
        }

        // Delete thumbnail
        if (!empty($activity['thumbnail'])) {
            delete_media($activity['thumbnail']);
        }

        if ($this->activityModel->delete($id)) {
            return redirect()->to('/admin/activities')->with('success', 'Aktivitas berhasil dihapus!');
        }

        return redirect()->to('/admin/activities')->with('error', 'Gagal menghapus aktivitas.');
    }
}
