<?php

namespace App\Models;

use CodeIgniter\Model;

class ActivityModel extends Model
{
    protected $table            = 'activities';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = true;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'title', 'slug', 'activity_date', 'category', 
        'tags_json', 'content', 'thumbnail', 'status'
    ];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules = [
        'title' => 'required|max_length[255]',
        'slug'  => 'required|max_length[255]|is_unique[activities.slug,id,{id}]',
    ];

    protected $validationMessages = [
        'title' => [
            'required' => 'Judul aktivitas wajib diisi.',
        ],
        'slug' => [
            'required'  => 'Slug wajib diisi.',
            'is_unique' => 'Slug sudah digunakan.',
        ],
    ];

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = ['generateSlug'];
    protected $beforeUpdate   = ['generateSlug'];

    /**
     * Auto-generate slug from title
     */
    protected function generateSlug(array $data): array
    {
        if (isset($data['data']['title']) && empty($data['data']['slug'])) {
            $data['data']['slug'] = url_title($data['data']['title'], '-', true);
        }
        return $data;
    }

    /**
     * Get published activities
     */
    public function getPublished(int $limit = 0, int $offset = 0): array
    {
        $builder = $this->where('status', 'publish')
                        ->orderBy('activity_date', 'DESC');
        
        if ($limit > 0) {
            $builder->limit($limit, $offset);
        }
        
        $results = $builder->findAll();
        return array_map([$this, 'decodeTags'], $results);
    }

    /**
     * Get activity by slug
     */
    public function getBySlug(string $slug): ?array
    {
        $result = $this->where('slug', $slug)
                       ->where('status', 'publish')
                       ->first();
        return $result ? $this->decodeTags($result) : null;
    }

    /**
     * Get activities by category
     */
    public function getByCategory(string $category): array
    {
        $results = $this->where('category', $category)
                        ->where('status', 'publish')
                        ->orderBy('activity_date', 'DESC')
                        ->findAll();
        return array_map([$this, 'decodeTags'], $results);
    }

    /**
     * Get all categories
     */
    public function getCategories(): array
    {
        return $this->select('category')
                    ->where('status', 'publish')
                    ->where('category IS NOT NULL')
                    ->groupBy('category')
                    ->findAll();
    }

    /**
     * Search activities
     */
    public function search(string $keyword): array
    {
        $results = $this->like('title', $keyword)
                        ->orLike('content', $keyword)
                        ->where('status', 'publish')
                        ->orderBy('activity_date', 'DESC')
                        ->findAll();
        return array_map([$this, 'decodeTags'], $results);
    }

    /**
     * Decode tags JSON
     */
    protected function decodeTags(array $item): array
    {
        $item['tags'] = json_decode($item['tags_json'] ?? '[]', true) ?: [];
        return $item;
    }
}
