<?php

namespace App\Controllers;

use App\Models\BiodataModel;
use App\Models\RiwayatPendidikanModel;
use App\Models\AktivitasHarianModel;

class Admin extends BaseController
{
    protected $biodataModel;
    protected $riwayatPendidikanModel;
    protected $aktivitasHarianModel;

    public function __construct()
    {
        $this->biodataModel = new BiodataModel();
        $this->riwayatPendidikanModel = new RiwayatPendidikanModel();
        $this->aktivitasHarianModel = new AktivitasHarianModel();
    }

    public function index()
    {
        $data = [
            'title' => 'Dashboard Admin',
            'biodata_count' => $this->biodataModel->countAll(),
            'pendidikan_count' => $this->riwayatPendidikanModel->countAll(),
            'aktivitas_count' => $this->aktivitasHarianModel->countAll(),
            'recent_activities' => $this->aktivitasHarianModel->orderBy('created_at', 'DESC')->limit(5)->findAll()
        ];

        return view('admin/dashboard', $data);
    }

    // BIODATA CRUD
    public function biodata()
    {
        $keyword = $this->request->getGet('search') ?? '';
        $orderBy = $this->request->getGet('sort') ?? 'id';
        $orderDir = $this->request->getGet('dir') ?? 'ASC';
        $page = $this->request->getGet('page') ?? 1;
        $perPage = 10;
        $offset = ($page - 1) * $perPage;

        $biodata = $this->biodataModel->search($keyword, $perPage, $offset, $orderBy, $orderDir);
        $total = $this->biodataModel->countSearch($keyword);

        $data = [
            'title' => 'Kelola Biodata',
            'biodata' => $biodata,
            'pager' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'total_pages' => ceil($total / $perPage)
            ],
            'search' => $keyword,
            'sort' => $orderBy,
            'dir' => $orderDir
        ];

        return view('admin/biodata/index', $data);
    }

    public function biodataCreate()
    {
        $data = ['title' => 'Tambah Biodata'];
        return view('admin/biodata/create', $data);
    }

    public function biodataStore()
    {
        if (!$this->validate($this->biodataModel->getValidationRules())) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = $this->request->getPost();
        
        // Handle file upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $newName = $foto->getRandomName();
            $foto->move(ROOTPATH . 'public/uploads/biodata', $newName);
            $data['foto'] = $newName;
        }

        if ($this->biodataModel->insert($data)) {
            return redirect()->to('/admin/biodata')->with('success', 'Data biodata berhasil ditambahkan');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan data biodata');
        }
    }

    public function biodataEdit($id)
    {
        $biodata = $this->biodataModel->find($id);
        if (!$biodata) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data biodata tidak ditemukan');
        }

        $data = [
            'title' => 'Edit Biodata',
            'biodata' => $biodata
        ];

        return view('admin/biodata/edit', $data);
    }

    public function biodataUpdate($id)
    {
        $biodata = $this->biodataModel->find($id);
        if (!$biodata) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data biodata tidak ditemukan');
        }

        if (!$this->validate($this->biodataModel->getValidationRules())) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = $this->request->getPost();
        
        // Handle file upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            // Delete old file
            if ($biodata['foto'] && file_exists(ROOTPATH . 'public/uploads/biodata/' . $biodata['foto'])) {
                unlink(ROOTPATH . 'public/uploads/biodata/' . $biodata['foto']);
            }
            
            $newName = $foto->getRandomName();
            $foto->move(ROOTPATH . 'public/uploads/biodata', $newName);
            $data['foto'] = $newName;
        }

        if ($this->biodataModel->update($id, $data)) {
            return redirect()->to('/admin/biodata')->with('success', 'Data biodata berhasil diperbarui');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal memperbarui data biodata');
        }
    }

    public function biodataDelete($id)
    {
        $biodata = $this->biodataModel->find($id);
        if (!$biodata) {
            return redirect()->to('/admin/biodata')->with('error', 'Data biodata tidak ditemukan');
        }

        // Delete file
        if ($biodata['foto'] && file_exists(ROOTPATH . 'public/uploads/biodata/' . $biodata['foto'])) {
            unlink(ROOTPATH . 'public/uploads/biodata/' . $biodata['foto']);
        }

        if ($this->biodataModel->delete($id)) {
            return redirect()->to('/admin/biodata')->with('success', 'Data biodata berhasil dihapus');
        } else {
            return redirect()->to('/admin/biodata')->with('error', 'Gagal menghapus data biodata');
        }
    }
}
    // PENDIDIKAN CRUD
    public function pendidikan()
    {
        $keyword = $this->request->getGet('search') ?? '';
        $jenjang = $this->request->getGet('jenjang') ?? '';
        $orderBy = $this->request->getGet('sort') ?? 'tahun_masuk';
        $orderDir = $this->request->getGet('dir') ?? 'DESC';
        $page = $this->request->getGet('page') ?? 1;
        $perPage = 10;
        $offset = ($page - 1) * $perPage;

        $pendidikan = $this->riwayatPendidikanModel->search($keyword, $jenjang, $perPage, $offset, $orderBy, $orderDir);
        $total = $this->riwayatPendidikanModel->countSearch($keyword, $jenjang);

        $data = [
            'title' => 'Kelola Riwayat Pendidikan',
            'pendidikan' => $pendidikan,
            'jenjang_options' => $this->riwayatPendidikanModel->getJenjangOptions(),
            'pager' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'total_pages' => ceil($total / $perPage)
            ],
            'search' => $keyword,
            'jenjang_filter' => $jenjang,
            'sort' => $orderBy,
            'dir' => $orderDir
        ];

        return view('admin/pendidikan/index', $data);
    }

    public function pendidikanCreate()
    {
        $data = [
            'title' => 'Tambah Riwayat Pendidikan',
            'jenjang_options' => $this->riwayatPendidikanModel->getJenjangOptions()
        ];
        return view('admin/pendidikan/create', $data);
    }

    public function pendidikanStore()
    {
        if (!$this->validate($this->riwayatPendidikanModel->getValidationRules())) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = $this->request->getPost();

        if ($this->riwayatPendidikanModel->insert($data)) {
            return redirect()->to('/admin/pendidikan')->with('success', 'Data riwayat pendidikan berhasil ditambahkan');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan data riwayat pendidikan');
        }
    }

    public function pendidikanEdit($id)
    {
        $pendidikan = $this->riwayatPendidikanModel->find($id);
        if (!$pendidikan) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data riwayat pendidikan tidak ditemukan');
        }

        $data = [
            'title' => 'Edit Riwayat Pendidikan',
            'pendidikan' => $pendidikan,
            'jenjang_options' => $this->riwayatPendidikanModel->getJenjangOptions()
        ];

        return view('admin/pendidikan/edit', $data);
    }

    public function pendidikanUpdate($id)
    {
        $pendidikan = $this->riwayatPendidikanModel->find($id);
        if (!$pendidikan) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data riwayat pendidikan tidak ditemukan');
        }

        if (!$this->validate($this->riwayatPendidikanModel->getValidationRules())) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = $this->request->getPost();

        if ($this->riwayatPendidikanModel->update($id, $data)) {
            return redirect()->to('/admin/pendidikan')->with('success', 'Data riwayat pendidikan berhasil diperbarui');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal memperbarui data riwayat pendidikan');
        }
    }

    public function pendidikanDelete($id)
    {
        $pendidikan = $this->riwayatPendidikanModel->find($id);
        if (!$pendidikan) {
            return redirect()->to('/admin/pendidikan')->with('error', 'Data riwayat pendidikan tidak ditemukan');
        }

        if ($this->riwayatPendidikanModel->delete($id)) {
            return redirect()->to('/admin/pendidikan')->with('success', 'Data riwayat pendidikan berhasil dihapus');
        } else {
            return redirect()->to('/admin/pendidikan')->with('error', 'Gagal menghapus data riwayat pendidikan');
        }
    }

    // AKTIVITAS CRUD
    public function aktivitas()
    {
        $keyword = $this->request->getGet('search') ?? '';
        $kategori = $this->request->getGet('kategori') ?? '';
        $tanggal_dari = $this->request->getGet('tanggal_dari') ?? '';
        $tanggal_sampai = $this->request->getGet('tanggal_sampai') ?? '';
        $orderBy = $this->request->getGet('sort') ?? 'tanggal';
        $orderDir = $this->request->getGet('dir') ?? 'DESC';
        $page = $this->request->getGet('page') ?? 1;
        $perPage = 10;
        $offset = ($page - 1) * $perPage;

        $aktivitas = $this->aktivitasHarianModel->search($keyword, $kategori, $tanggal_dari, $tanggal_sampai, $perPage, $offset, $orderBy, $orderDir);
        $total = $this->aktivitasHarianModel->countSearch($keyword, $kategori, $tanggal_dari, $tanggal_sampai);
        $kategori_list = $this->aktivitasHarianModel->getKategoriList();

        $data = [
            'title' => 'Kelola Aktivitas Harian',
            'aktivitas' => $aktivitas,
            'kategori_list' => $kategori_list,
            'pager' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'total_pages' => ceil($total / $perPage)
            ],
            'search' => $keyword,
            'kategori_filter' => $kategori,
            'tanggal_dari' => $tanggal_dari,
            'tanggal_sampai' => $tanggal_sampai,
            'sort' => $orderBy,
            'dir' => $orderDir
        ];

        return view('admin/aktivitas/index', $data);
    }

    public function aktivitasCreate()
    {
        $data = ['title' => 'Tambah Aktivitas Harian'];
        return view('admin/aktivitas/create', $data);
    }

    public function aktivitasStore()
    {
        if (!$this->validate($this->aktivitasHarianModel->getValidationRules())) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = $this->request->getPost();
        
        // Handle file uploads
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $newName = $foto->getRandomName();
            $foto->move(ROOTPATH . 'public/uploads/aktivitas', $newName);
            $data['foto'] = $newName;
        }

        $video = $this->request->getFile('video');
        if ($video && $video->isValid() && !$video->hasMoved()) {
            $newName = $video->getRandomName();
            $video->move(ROOTPATH . 'public/uploads/aktivitas', $newName);
            $data['video'] = $newName;
        }

        if ($this->aktivitasHarianModel->insert($data)) {
            return redirect()->to('/admin/aktivitas')->with('success', 'Data aktivitas harian berhasil ditambahkan');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan data aktivitas harian');
        }
    }

    public function aktivitasEdit($id)
    {
        $aktivitas = $this->aktivitasHarianModel->find($id);
        if (!$aktivitas) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data aktivitas harian tidak ditemukan');
        }

        $data = [
            'title' => 'Edit Aktivitas Harian',
            'aktivitas' => $aktivitas
        ];

        return view('admin/aktivitas/edit', $data);
    }

    public function aktivitasUpdate($id)
    {
        $aktivitas = $this->aktivitasHarianModel->find($id);
        if (!$aktivitas) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data aktivitas harian tidak ditemukan');
        }

        if (!$this->validate($this->aktivitasHarianModel->getValidationRules())) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = $this->request->getPost();
        
        // Handle file uploads
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            // Delete old file
            if ($aktivitas['foto'] && file_exists(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['foto'])) {
                unlink(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['foto']);
            }
            
            $newName = $foto->getRandomName();
            $foto->move(ROOTPATH . 'public/uploads/aktivitas', $newName);
            $data['foto'] = $newName;
        }

        $video = $this->request->getFile('video');
        if ($video && $video->isValid() && !$video->hasMoved()) {
            // Delete old file
            if ($aktivitas['video'] && file_exists(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['video'])) {
                unlink(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['video']);
            }
            
            $newName = $video->getRandomName();
            $video->move(ROOTPATH . 'public/uploads/aktivitas', $newName);
            $data['video'] = $newName;
        }

        if ($this->aktivitasHarianModel->update($id, $data)) {
            return redirect()->to('/admin/aktivitas')->with('success', 'Data aktivitas harian berhasil diperbarui');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal memperbarui data aktivitas harian');
        }
    }

    public function aktivitasDelete($id)
    {
        $aktivitas = $this->aktivitasHarianModel->find($id);
        if (!$aktivitas) {
            return redirect()->to('/admin/aktivitas')->with('error', 'Data aktivitas harian tidak ditemukan');
        }

        // Delete files
        if ($aktivitas['foto'] && file_exists(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['foto'])) {
            unlink(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['foto']);
        }
        if ($aktivitas['video'] && file_exists(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['video'])) {
            unlink(ROOTPATH . 'public/uploads/aktivitas/' . $aktivitas['video']);
        }

        if ($this->aktivitasHarianModel->delete($id)) {
            return redirect()->to('/admin/aktivitas')->with('success', 'Data aktivitas harian berhasil dihapus');
        } else {
            return redirect()->to('/admin/aktivitas')->with('error', 'Gagal menghapus data aktivitas harian');
        }
    }
}