<?php

namespace App\Models;

use CodeIgniter\Model;

class RiwayatPendidikanModel extends Model
{
    protected $table = 'riwayat_pendidikan';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'jenjang', 'nama_sekolah', 'jurusan', 'tahun_masuk', 
        'tahun_lulus', 'nilai_akhir', 'prestasi', 'alamat_sekolah'
    ];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    protected $validationRules = [
        'jenjang' => 'required|in_list[SD,SMP,SMA,D3,S1,S2,S3]',
        'nama_sekolah' => 'required|min_length[3]|max_length[200]',
        'jurusan' => 'permit_empty|max_length[100]',
        'tahun_masuk' => 'required|integer|greater_than[1900]',
        'tahun_lulus' => 'permit_empty|integer|greater_than_equal_to[{tahun_masuk}]',
        'nilai_akhir' => 'permit_empty|max_length[10]',
        'prestasi' => 'permit_empty',
        'alamat_sekolah' => 'permit_empty'
    ];

    protected $validationMessages = [
        'jenjang' => [
            'required' => 'Jenjang pendidikan harus dipilih',
            'in_list' => 'Jenjang pendidikan tidak valid'
        ],
        'nama_sekolah' => [
            'required' => 'Nama sekolah harus diisi',
            'min_length' => 'Nama sekolah minimal 3 karakter',
            'max_length' => 'Nama sekolah maksimal 200 karakter'
        ],
        'tahun_masuk' => [
            'required' => 'Tahun masuk harus diisi',
            'integer' => 'Tahun masuk harus berupa angka',
            'greater_than' => 'Tahun masuk tidak valid'
        ]
    ];

    public function search($keyword = '', $jenjang = '', $limit = 10, $offset = 0, $orderBy = 'tahun_masuk', $orderDir = 'DESC')
    {
        $builder = $this->builder();
        
        if (!empty($keyword)) {
            $builder->groupStart()
                    ->like('nama_sekolah', $keyword)
                    ->orLike('jurusan', $keyword)
                    ->orLike('prestasi', $keyword)
                    ->groupEnd();
        }
        
        if (!empty($jenjang)) {
            $builder->where('jenjang', $jenjang);
        }
        
        return $builder->orderBy($orderBy, $orderDir)
                      ->limit($limit, $offset)
                      ->get()
                      ->getResultArray();
    }

    public function countSearch($keyword = '', $jenjang = '')
    {
        $builder = $this->builder();
        
        if (!empty($keyword)) {
            $builder->groupStart()
                    ->like('nama_sekolah', $keyword)
                    ->orLike('jurusan', $keyword)
                    ->orLike('prestasi', $keyword)
                    ->groupEnd();
        }
        
        if (!empty($jenjang)) {
            $builder->where('jenjang', $jenjang);
        }
        
        return $builder->countAllResults();
    }

    public function getJenjangOptions()
    {
        return [
            'SD' => 'Sekolah Dasar',
            'SMP' => 'Sekolah Menengah Pertama',
            'SMA' => 'Sekolah Menengah Atas',
            'D3' => 'Diploma 3',
            'S1' => 'Sarjana (S1)',
            'S2' => 'Magister (S2)',
            'S3' => 'Doktor (S3)'
        ];
    }
}