<?php

namespace App\Controllers;

use App\Models\BiodataModel;
use App\Models\PendidikanModel;
use App\Models\AktivitasModel;

class Admin extends BaseController
{
    protected $biodataModel;
    protected $pendidikanModel;
    protected $aktivitasModel;
    protected $session;
    protected $adminUsername;
    protected $adminPasswordHash;

    public function __construct()
    {
        $this->biodataModel = new BiodataModel();
        $this->pendidikanModel = new PendidikanModel();
        $this->aktivitasModel = new AktivitasModel();
        $this->session = \Config\Services::session();
        
        // Ambil kredensial admin dari .env (dukung format dot/underscore)
        $this->adminUsername = env('admin.username', getenv('ADMIN_USERNAME') ?: 'admin');
        $adminPasswordHash   = env('admin.password_hash', getenv('ADMIN_PASSWORD_HASH'));
        
        // Hash default untuk password: Admin#2025 (hash statis, jangan diganti-ganti)
        $this->adminPasswordHash = $adminPasswordHash ?: '$2y$10$eNMhYdMfIPSnw1vrdh9RaOqKAwrKB8TpQ/CWH8dQw2y2.LT4DJdt.';
    }

    protected function decodeMediaPaths(?string $filePath): array
    {
        if (empty($filePath)) {
            return [];
        }

        $decoded = json_decode($filePath, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return array_values(array_filter($decoded));
        }

        return [$filePath];
    }

    // Check if user is logged in
    protected function checkLogin()
    {
        $isLoggedIn = $this->session->get('admin_logged_in');
        if (!$isLoggedIn) {
            return redirect()->to(base_url('admin/login'));
        }
        return null;
    }

    // Login
    public function login()
    {
        // Normalisasi method ke lowercase supaya perbandingan konsisten
        $method = strtolower($this->request->getMethod());
        $firstBiodata = $this->biodataModel->first();
        $previewFoto = $firstBiodata['foto'] ?? null;
        $data = [
            'title'       => 'Login Admin',
            'previewFoto' => $previewFoto,
        ];

        // Debug: log setiap request ke login
        log_message('debug', 'Login method called - Method: {method}', ['method' => $method]);
        
        // Jika sudah login, redirect ke dashboard
        if ($this->session->has('admin_logged_in') && $this->session->get('admin_logged_in')) {
            log_message('debug', 'Already logged in, redirecting to admin');
            return redirect()->to(base_url('admin'));
        }

        if ($method === 'post') {
            log_message('debug', 'POST request received to login');
            // Throttle brute-force attempts (5 tries -> 5 minute lock)
            $lockedUntil = $this->session->get('admin_lock_until');
            if ($lockedUntil && $lockedUntil > time()) {
                $data['error'] = 'Terlalu banyak percobaan login. Coba lagi setelah beberapa menit.';
                return view('admin/login', $data);
            }

            $username = $this->request->getPost('username');
            $password = $this->request->getPost('password');

            // Debug: cek username dan password
            $usernameMatch = ($username === $this->adminUsername);
            $passwordMatch = password_verify($password, $this->adminPasswordHash);
            $isValid = $usernameMatch && $passwordMatch;
            
            log_message('info', 'Admin login attempt user={user} username_match={umatch} password_match={pmatch} valid={valid}', [
                'user'  => $username,
                'umatch' => $usernameMatch ? 'yes' : 'no',
                'pmatch' => $passwordMatch ? 'yes' : 'no',
                'valid' => $isValid ? 'yes' : 'no',
            ]);

            if ($isValid) {
                log_message('info', 'Login credentials valid, setting session');
                
                // Set session
                $this->session->set('admin_logged_in', true);
                $this->session->remove('admin_attempts');
                $this->session->remove('admin_lock_until');
                
                // Verifikasi session sudah tersimpan
                $checkSession = $this->session->get('admin_logged_in');
                log_message('info', 'Login successful - Session set: {result}, redirecting to admin', ['result' => $checkSession ? 'true' : 'false']);
                
                // Redirect ke panel admin (dashboard) - gunakan header langsung untuk memastikan
                $redirectUrl = base_url('admin');
                log_message('debug', 'Redirect URL: {url}', ['url' => $redirectUrl]);
                
                return redirect()->to($redirectUrl)->with('login_success', 'Login berhasil!');
            } else {
                log_message('info', 'Login failed - Invalid credentials');
                $attempts = (int) $this->session->get('admin_attempts');
                $attempts++;
                $this->session->set('admin_attempts', $attempts);

                if ($attempts >= 5) {
                    $this->session->set('admin_lock_until', time() + 300); // 5 menit
                }

                $data['error'] = 'Username atau password salah!';
            }
        }

        return view('admin/login', $data);
    }

    // Logout
    public function logout()
    {
        $this->session->destroy();
        return redirect()->to(base_url('admin/login'));
    }

    // Dashboard
    public function index()
    {
        log_message('debug', 'Admin index called - checking login');
        $check = $this->checkLogin();
        if ($check !== null) {
            log_message('debug', 'Not logged in, redirecting to login');
            return $check;
        }
        
        log_message('debug', 'Logged in, showing dashboard');
        $data = [
            'title' => 'Dashboard Admin',
            'biodata_count' => $this->biodataModel->countAllResults(),
            'pendidikan_count' => $this->pendidikanModel->countAllResults(),
            'aktivitas_count' => $this->aktivitasModel->countAllResults()
        ];
        return view('admin/dashboard', $data);
    }

    // Biodata CRUD
    public function biodata()
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }
        
        $perPage     = 10;
        $search      = trim((string) $this->request->getGet('search'));
        $filterField = $this->request->getGet('filter_field');
        $filterValue = trim((string) $this->request->getGet('filter_value'));
        $sortBy      = $this->request->getGet('sort_by');
        $sortDir     = strtolower((string) $this->request->getGet('sort_dir')) === 'asc' ? 'asc' : 'desc';

        $allowedFilters = [
            'nama'        => 'nama',
            'email'       => 'email',
            'telepon'     => 'telepon',
            'keahlian'    => 'keahlian',
            'pengalaman'  => 'pengalaman',
        ];

        $allowedSorts = [
            'id'         => 'id',
            'nama'       => 'nama',
            'email'      => 'email',
            'telepon'    => 'telepon',
            'created_at' => 'created_at',
        ];

        if (! array_key_exists($filterField, $allowedFilters)) {
            $filterField = '';
        }

        if (! array_key_exists($sortBy, $allowedSorts)) {
            $sortBy = 'created_at';
        }

        $query = $this->biodataModel;

        if ($search !== '') {
            $query = $query->groupStart()
                ->like('nama', $search)
                ->orLike('email', $search)
                ->orLike('telepon', $search)
                ->orLike('keahlian', $search)
                ->orLike('pengalaman', $search)
                ->groupEnd();
        }

        if ($filterValue !== '' && $filterField !== '') {
            $query = $query->like($allowedFilters[$filterField], $filterValue);
        }

        $query = $query->orderBy($allowedSorts[$sortBy], $sortDir);

        $data = [
            'title'        => 'Kelola Biodata',
            'biodata'      => $query->paginate($perPage, 'biodata'),
            'pager'        => $this->biodataModel->pager,
            'search'       => $search,
            'filterField'  => $filterField,
            'filterValue'  => $filterValue,
            'sortBy'       => $sortBy,
            'sortDir'      => $sortDir,
            'perPage'      => $perPage,
            'pageParam'    => 'page_biodata',
        ];
        return view('admin/biodata/index', $data);
    }

    public function biodata_create()
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }

        $method = strtolower($this->request->getMethod());
        
        if ($method === 'post') {
            $fotoPath = $this->request->getPost('foto');
            $upload   = $this->request->getFile('foto_file');

            if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                $targetDir = FCPATH . 'uploads/biodata';
                if (! is_dir($targetDir)) {
                    mkdir($targetDir, 0755, true);
                }

                $newName  = $upload->getRandomName();
                $upload->move($targetDir, $newName);
                $fotoPath = 'uploads/biodata/' . $newName;
            }

            $data = [
                'nama' => $this->request->getPost('nama'),
                'alamat' => $this->request->getPost('alamat'),
                'email' => $this->request->getPost('email'),
                'telepon' => $this->request->getPost('telepon'),
                'foto' => $fotoPath,
                'tentang' => $this->request->getPost('tentang'),
                'keahlian' => $this->request->getPost('keahlian'),
                'pengalaman' => $this->request->getPost('pengalaman')
            ];
            
            $this->biodataModel->insert($data);
            return redirect()->to(base_url('admin/biodata'))->with('success', 'Biodata berhasil ditambahkan!');
        }

        $data['title'] = 'Tambah Biodata';
        return view('admin/biodata/create', $data);
    }

    public function biodata_edit($id)
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }

        $existing = $this->biodataModel->find($id);
        if (! $existing) {
            return redirect()->to(base_url('admin/biodata'))->with('error', 'Data tidak ditemukan');
        }

        $method = strtolower($this->request->getMethod());
        
        if ($method === 'post') {
            $fotoPath = $this->request->getPost('foto') ?: $existing['foto'];
            $upload   = $this->request->getFile('foto_file');

            if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                $targetDir = FCPATH . 'uploads/biodata';
                if (! is_dir($targetDir)) {
                    mkdir($targetDir, 0755, true);
                }

                $newName  = $upload->getRandomName();
                $upload->move($targetDir, $newName);
                $fotoPath = 'uploads/biodata/' . $newName;

                // Hapus foto lama jika ada dan berada di direktori uploads
                if (! empty($existing['foto'])) {
                    $oldFile = FCPATH . ltrim($existing['foto'], '/');
                    if (is_file($oldFile)) {
                        @unlink($oldFile);
                    }
                }
            }

            $data = [
                'nama' => $this->request->getPost('nama'),
                'alamat' => $this->request->getPost('alamat'),
                'email' => $this->request->getPost('email'),
                'telepon' => $this->request->getPost('telepon'),
                'foto' => $fotoPath,
                'tentang' => $this->request->getPost('tentang'),
                'keahlian' => $this->request->getPost('keahlian'),
                'pengalaman' => $this->request->getPost('pengalaman')
            ];
            
            $this->biodataModel->update($id, $data);
            return redirect()->to(base_url('admin/biodata'))->with('success', 'Biodata berhasil diupdate!');
        }

        $data = [
            'title' => 'Edit Biodata',
            'biodata' => $existing
        ];
        return view('admin/biodata/edit', $data);
    }

    public function biodata_delete($id)
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }
        if (!$this->request->is('post')) {
            return redirect()->back()->with('error', 'Metode tidak diizinkan.');
        }
        
        $this->biodataModel->delete($id);
        return redirect()->to(base_url('admin/biodata'))->with('success', 'Biodata berhasil dihapus!');
    }

    // Pendidikan CRUD
    public function pendidikan()
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }
        
        $perPage     = 10;
        $search      = trim((string) $this->request->getGet('search'));
        $filterField = $this->request->getGet('filter_field');
        $filterValue = trim((string) $this->request->getGet('filter_value'));
        $sortBy      = $this->request->getGet('sort_by');
        $sortDir     = strtolower((string) $this->request->getGet('sort_dir')) === 'asc' ? 'asc' : 'desc';

        $allowedFilters = [
            'jenjang'       => 'jenjang',
            'institusi'     => 'institusi',
            'jurusan'       => 'jurusan',
            'tahun_mulai'   => 'tahun_mulai',
            'tahun_selesai' => 'tahun_selesai',
        ];

        $allowedSorts = [
            'id'            => 'id',
            'jenjang'       => 'jenjang',
            'institusi'     => 'institusi',
            'jurusan'       => 'jurusan',
            'tahun_mulai'   => 'tahun_mulai',
            'tahun_selesai' => 'tahun_selesai',
            'created_at'    => 'created_at',
        ];

        if (! array_key_exists($filterField, $allowedFilters)) {
            $filterField = '';
        }

        if (! array_key_exists($sortBy, $allowedSorts)) {
            $sortBy = 'tahun_selesai';
        }

        $query = $this->pendidikanModel;

        if ($search !== '') {
            $query = $query->groupStart()
                ->like('jenjang', $search)
                ->orLike('institusi', $search)
                ->orLike('jurusan', $search)
                ->orLike('tahun_mulai', $search)
                ->orLike('tahun_selesai', $search)
                ->orLike('deskripsi', $search)
                ->groupEnd();
        }

        if ($filterValue !== '' && $filterField !== '') {
            $query = $query->like($allowedFilters[$filterField], $filterValue);
        }

        $query = $query->orderBy($allowedSorts[$sortBy], $sortDir);

        $data = [
            'title'       => 'Kelola Pendidikan',
            'pendidikan'  => $query->paginate($perPage, 'pendidikan'),
            'pager'       => $this->pendidikanModel->pager,
            'search'      => $search,
            'filterField' => $filterField,
            'filterValue' => $filterValue,
            'sortBy'      => $sortBy,
            'sortDir'     => $sortDir,
            'perPage'     => $perPage,
            'pageParam'   => 'page_pendidikan',
        ];
        return view('admin/pendidikan/index', $data);
    }

    public function pendidikan_create()
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }

        $method = strtolower($this->request->getMethod());
        
        if ($method === 'post') {
            $fotoPath = null;
            $upload   = $this->request->getFile('foto_file');

            if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                $targetDir = FCPATH . 'uploads/pendidikan';
                if (! is_dir($targetDir)) {
                    mkdir($targetDir, 0755, true);
                }

                $newName = $upload->getRandomName();
                $upload->move($targetDir, $newName);
                $fotoPath = 'uploads/pendidikan/' . $newName;
            }

            $data = [
                'jenjang' => $this->request->getPost('jenjang'),
                'institusi' => $this->request->getPost('institusi'),
                'jurusan' => $this->request->getPost('jurusan'),
                'tahun_mulai' => $this->request->getPost('tahun_mulai'),
                'tahun_selesai' => $this->request->getPost('tahun_selesai'),
                'deskripsi' => $this->request->getPost('deskripsi'),
                'foto' => $fotoPath,
            ];
            
            $this->pendidikanModel->insert($data);
            return redirect()->to(base_url('admin/pendidikan'))->with('success', 'Pendidikan berhasil ditambahkan!');
        }

        $data['title'] = 'Tambah Pendidikan';
        return view('admin/pendidikan/create', $data);
    }

    public function pendidikan_edit($id)
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }

        $existing = $this->pendidikanModel->find($id);
        if (! $existing) {
            return redirect()->to(base_url('admin/pendidikan'))->with('error', 'Data tidak ditemukan');
        }

        $method = strtolower($this->request->getMethod());
        
        if ($method === 'post') {
            $fotoPath = $this->request->getPost('foto') ?: ($existing['foto'] ?? null);
            $upload   = $this->request->getFile('foto_file');

            if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                $targetDir = FCPATH . 'uploads/pendidikan';
                if (! is_dir($targetDir)) {
                    mkdir($targetDir, 0755, true);
                }

                $newName = $upload->getRandomName();
                $upload->move($targetDir, $newName);
                $fotoPath = 'uploads/pendidikan/' . $newName;

                // Hapus foto lama jika ada
                if (! empty($existing['foto'])) {
                    $oldFile = FCPATH . ltrim($existing['foto'], '/');
                    if (is_file($oldFile)) {
                        @unlink($oldFile);
                    }
                }
            }

            $data = [
                'jenjang' => $this->request->getPost('jenjang'),
                'institusi' => $this->request->getPost('institusi'),
                'jurusan' => $this->request->getPost('jurusan'),
                'tahun_mulai' => $this->request->getPost('tahun_mulai'),
                'tahun_selesai' => $this->request->getPost('tahun_selesai'),
                'deskripsi' => $this->request->getPost('deskripsi'),
                'foto' => $fotoPath,
            ];
            
            $this->pendidikanModel->update($id, $data);
            return redirect()->to(base_url('admin/pendidikan'))->with('success', 'Pendidikan berhasil diupdate!');
        }

        $data = [
            'title' => 'Edit Pendidikan',
            'pendidikan' => $existing
        ];
        return view('admin/pendidikan/edit', $data);
    }

    public function pendidikan_delete($id)
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }
        if (!$this->request->is('post')) {
            return redirect()->back()->with('error', 'Metode tidak diizinkan.');
        }
        
        $id = (int) $id;
        if ($id <= 0) {
            return redirect()->back()->with('error', 'Data tidak valid.');
        }

        $existing = $this->pendidikanModel->find($id);
        if (! $existing) {
            return redirect()->to(base_url('admin/pendidikan'))->with('error', 'Data tidak ditemukan.');
        }

        if (! empty($existing['foto'])) {
            $oldFile = FCPATH . ltrim($existing['foto'], '/');
            if (is_file($oldFile)) {
                @unlink($oldFile);
            }
        }

        $this->pendidikanModel->delete($id);
        return redirect()->to(base_url('admin/pendidikan'))->with('success', 'Pendidikan berhasil dihapus!');
    }

    // Aktivitas CRUD
    public function aktivitas()
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }
        
        $perPage     = 10;
        $search      = trim((string) $this->request->getGet('search'));
        $filterField = $this->request->getGet('filter_field');
        $filterValue = trim((string) $this->request->getGet('filter_value'));
        $sortBy      = $this->request->getGet('sort_by');
        $sortDir     = strtolower((string) $this->request->getGet('sort_dir')) === 'asc' ? 'asc' : 'desc';

        $allowedFilters = [
            'nama_aktivitas' => 'nama_aktivitas',
            'jenis'          => 'jenis',
            'tanggal'        => 'tanggal',
        ];

        $allowedSorts = [
            'id'             => 'id',
            'nama_aktivitas' => 'nama_aktivitas',
            'tanggal'        => 'tanggal',
            'jam'            => 'jam',
            'jenis'          => 'jenis',
            'created_at'     => 'created_at',
        ];

        if (! array_key_exists($filterField, $allowedFilters)) {
            $filterField = '';
        }

        if (! array_key_exists($sortBy, $allowedSorts)) {
            $sortBy = 'tanggal';
        }

        $query = $this->aktivitasModel;

        if ($search !== '') {
            $query = $query->groupStart()
                ->like('nama_aktivitas', $search)
                ->orLike('deskripsi', $search)
                ->orLike('jenis', $search)
                ->orLike('tanggal', $search)
                ->orLike('jam', $search)
                ->groupEnd();
        }

        if ($filterValue !== '' && $filterField !== '') {
            $query = $query->like($allowedFilters[$filterField], $filterValue);
        }

        $query = $query->orderBy($allowedSorts[$sortBy], $sortDir);

        if ($sortBy === 'tanggal') {
            // Konsistenkan urutan tanggal dengan jam
            $query = $query->orderBy('jam', $sortDir);
        }

        $data = [
            'title'       => 'Kelola Aktivitas',
            'aktivitas'   => $query->paginate($perPage, 'aktivitas'),
            'pager'       => $this->aktivitasModel->pager,
            'search'      => $search,
            'filterField' => $filterField,
            'filterValue' => $filterValue,
            'sortBy'      => $sortBy,
            'sortDir'     => $sortDir,
            'perPage'     => $perPage,
            'pageParam'   => 'page_aktivitas',
        ];
        return view('admin/aktivitas/index', $data);
    }

    public function aktivitas_create()
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }

        $method = strtolower($this->request->getMethod());
        
        if ($method === 'post') {
            $filePath = null;
            $jenis    = $this->request->getPost('jenis');

            if ($jenis === 'foto') {
                $photoCount = (int) $this->request->getPost('photo_count');
                $photoCount = $photoCount < 2 ? 2 : ($photoCount > 4 ? 4 : $photoCount);

                $uploads     = $this->request->getFileMultiple('photo_uploads');
                $savedPhotos = [];

                if ($uploads) {
                    $targetDir = FCPATH . 'uploads/aktivitas';
                    if (! is_dir($targetDir)) {
                        mkdir($targetDir, 0755, true);
                    }

                    foreach ($uploads as $upload) {
                        if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                            $newName = $upload->getRandomName();
                            $upload->move($targetDir, $newName);
                            $savedPhotos[] = 'uploads/aktivitas/' . $newName;
                        }
                    }
                }

                if (count($savedPhotos) < $photoCount) {
                    return redirect()->back()->withInput()->with('error', 'Unggah minimal ' . $photoCount . ' foto.');
                }

                if (! empty($savedPhotos)) {
                    $filePath = json_encode(array_slice($savedPhotos, 0, 4));
                }
            } else {
                $upload = $this->request->getFile('file_upload');

                if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                    $targetDir = FCPATH . 'uploads/aktivitas';
                    if (! is_dir($targetDir)) {
                        mkdir($targetDir, 0755, true);
                    }

                    $newName = $upload->getRandomName();
                    $upload->move($targetDir, $newName);
                    $filePath = 'uploads/aktivitas/' . $newName;
                }
            }

            $data = [
                'tanggal' => $this->request->getPost('tanggal'),
                'jam' => $this->request->getPost('jam'),
                'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
                'deskripsi' => $this->request->getPost('deskripsi'),
                'jenis' => $jenis,
                'file_path' => $filePath
            ];
            
            $this->aktivitasModel->insert($data);
            return redirect()->to(base_url('admin/aktivitas'))->with('success', 'Aktivitas berhasil ditambahkan!');
        }

        $data['title'] = 'Tambah Aktivitas';
        return view('admin/aktivitas/create', $data);
    }

    public function aktivitas_edit($id)
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }

        $existing = $this->aktivitasModel->find($id);
        if (! $existing) {
            return redirect()->to(base_url('admin/aktivitas'))->with('error', 'Data tidak ditemukan');
        }

        $method = strtolower($this->request->getMethod());
        
        if ($method === 'post') {
            $filePath      = $existing['file_path'];
            $existingMedia = $this->decodeMediaPaths($existing['file_path']);
            $jenis         = $this->request->getPost('jenis');

            if ($jenis === 'foto') {
                $photoCount = (int) $this->request->getPost('photo_count');
                $photoCount = $photoCount < 2 ? 2 : ($photoCount > 4 ? 4 : $photoCount);

                $uploads     = $this->request->getFileMultiple('photo_uploads');
                $savedPhotos = [];

                if ($uploads) {
                    $targetDir = FCPATH . 'uploads/aktivitas';
                    if (! is_dir($targetDir)) {
                        mkdir($targetDir, 0755, true);
                    }

                    foreach ($uploads as $upload) {
                        if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                            $newName = $upload->getRandomName();
                            $upload->move($targetDir, $newName);
                            $savedPhotos[] = 'uploads/aktivitas/' . $newName;
                        }
                    }
                }

                if (! empty($savedPhotos)) {
                    foreach ($existingMedia as $old) {
                        $oldFile = FCPATH . ltrim($old, '/');
                        if (is_file($oldFile)) {
                            @unlink($oldFile);
                        }
                    }
                    $filePath = json_encode(array_slice($savedPhotos, 0, 4));
                } elseif ($existing['jenis'] === 'video') {
                    return redirect()->back()->withInput()->with('error', 'Unggah minimal ' . $photoCount . ' foto saat mengganti ke jenis foto.');
                }
            } else {
                $upload = $this->request->getFile('file_upload');

                if ($upload && $upload->isValid() && ! $upload->hasMoved()) {
                    $targetDir = FCPATH . 'uploads/aktivitas';
                    if (! is_dir($targetDir)) {
                        mkdir($targetDir, 0755, true);
                    }

                    $newName = $upload->getRandomName();
                    $upload->move($targetDir, $newName);
                    $filePath = 'uploads/aktivitas/' . $newName;

                    // Hapus file lama jika ada (baik foto maupun video)
                    foreach ($existingMedia as $old) {
                        $oldFile = FCPATH . ltrim($old, '/');
                        if (is_file($oldFile)) {
                            @unlink($oldFile);
                        }
                    }
                }
            }

            $data = [
                'tanggal' => $this->request->getPost('tanggal'),
                'jam' => $this->request->getPost('jam'),
                'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
                'deskripsi' => $this->request->getPost('deskripsi'),
                'jenis' => $jenis,
                'file_path' => $filePath
            ];
            
            $this->aktivitasModel->update($id, $data);
            return redirect()->to(base_url('admin/aktivitas'))->with('success', 'Aktivitas berhasil diupdate!');
        }

        $data = [
            'title' => 'Edit Aktivitas',
            'aktivitas' => $existing
        ];
        return view('admin/aktivitas/edit', $data);
    }

    public function aktivitas_delete($id)
    {
        $check = $this->checkLogin();
        if ($check !== null) {
            return $check;
        }
        if (! $this->request->is('post')) {
            return redirect()->back()->with('error', 'Metode tidak diizinkan.');
        }
        
        $id = (int) $id;
        if ($id <= 0) {
            return redirect()->back()->with('error', 'Data tidak valid.');
        }

        $existing = $this->aktivitasModel->find($id);
        if (! $existing) {
            return redirect()->to(base_url('admin/aktivitas'))->with('error', 'Data tidak ditemukan.');
        }

        // Hapus file media terkait (foto/video)
        $mediaFiles = $this->decodeMediaPaths($existing['file_path'] ?? null);
        foreach ($mediaFiles as $file) {
            $filePath = FCPATH . ltrim($file, '/');
            if (is_file($filePath)) {
                @unlink($filePath);
            }
        }

        $this->aktivitasModel->delete($id);
        return redirect()->to(base_url('admin/aktivitas'))->with('success', 'Aktivitas berhasil dihapus!');
    }
}
