<?php

namespace App\Controllers;

use App\Models\BiodataModel;
use App\Models\PendidikanModel;
use App\Models\AktivitasModel;

class Home extends BaseController
{
    protected $biodataModel;
    protected $pendidikanModel;
    protected $aktivitasModel;

    public function __construct()
    {
        $this->biodataModel = new BiodataModel();
        $this->pendidikanModel = new PendidikanModel();
        $this->aktivitasModel = new AktivitasModel();
    }

    private function normalizeAktivitasMedia(array $item): array
    {
        $media = [];

        if (! empty($item['file_path'])) {
            $decoded = json_decode($item['file_path'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $media = array_values(array_filter($decoded));
            } else {
                $media = [$item['file_path']];
            }
        }

        $item['media_files'] = $media;
        $item['cover_path']  = $media[0] ?? null;

        return $item;
    }

    public function index(): string
    {
        $biodata = $this->biodataModel->first();

        $data = [
            'title' => 'Portfolio Ridwan',
            'biodata' => $biodata,
            'pendidikanCount' => $this->pendidikanModel->countAllResults(),
            'aktivitasCount' => $this->aktivitasModel->countAllResults(),
            'latestAktivitas' => array_map(
                fn ($item) => $this->normalizeAktivitasMedia($item),
                $this->aktivitasModel
                    ->orderBy('tanggal', 'DESC')
                    ->orderBy('jam', 'DESC')
                    ->findAll(3)
            ),
        ];
        return view('home', $data);
    }

    public function cv(): string
    {
        $data = [
            'title' => 'Biodata - Ridwan',
            'biodata' => $this->biodataModel->first()
        ];
        return view('cv', $data);
    }

    public function pendidikan(): string
    {
        $perPage     = 10;
        $search      = trim((string) $this->request->getGet('search'));
        $filterField = $this->request->getGet('filter_field');
        $filterValue = trim((string) $this->request->getGet('filter_value'));
        $sortBy      = $this->request->getGet('sort_by');
        $sortDir     = strtolower((string) $this->request->getGet('sort_dir')) === 'asc' ? 'asc' : 'desc';
        $pageParam   = 'page_pendidikan';
        $page        = (int) ($this->request->getGet($pageParam) ?? 1);

        $allowedFilters = [
            'jenjang'       => 'jenjang',
            'institusi'     => 'institusi',
            'jurusan'       => 'jurusan',
            'tahun_mulai'   => 'tahun_mulai',
            'tahun_selesai' => 'tahun_selesai',
        ];

        $allowedSorts = [
            'id'            => 'id',
            'jenjang'       => 'jenjang',
            'institusi'     => 'institusi',
            'jurusan'       => 'jurusan',
            'tahun_mulai'   => 'tahun_mulai',
            'tahun_selesai' => 'tahun_selesai',
            'created_at'    => 'created_at',
        ];

        if (! array_key_exists($filterField, $allowedFilters)) {
            $filterField = '';
        }

        if (! array_key_exists($sortBy, $allowedSorts)) {
            $sortBy = 'tahun_selesai';
        }

        $query = $this->pendidikanModel;

        if ($search !== '') {
            $query = $query->groupStart()
                ->like('jenjang', $search)
                ->orLike('institusi', $search)
                ->orLike('jurusan', $search)
                ->orLike('tahun_mulai', $search)
                ->orLike('tahun_selesai', $search)
                ->orLike('deskripsi', $search)
                ->groupEnd();
        }

        if ($filterValue !== '' && $filterField !== '') {
            $query = $query->like($allowedFilters[$filterField], $filterValue);
        }

        $query = $query->orderBy($allowedSorts[$sortBy], $sortDir);

        $data = [
            'title'        => 'Pendidikan - Ridwan',
            'pendidikan'   => $query->paginate($perPage, 'pendidikan', $page),
            'pager'        => $this->pendidikanModel->pager,
            'search'       => $search,
            'filterField'  => $filterField,
            'filterValue'  => $filterValue,
            'sortBy'       => $sortBy,
            'sortDir'      => $sortDir,
            'perPage'      => $perPage,
            'pageParam'    => $pageParam,
        ];
        return view('pendidikan', $data);
    }

    public function aktivitas(): string
    {
        $perPage     = 10;
        $search      = trim((string) $this->request->getGet('search'));
        $filterField = $this->request->getGet('filter_field');
        $filterValue = trim((string) $this->request->getGet('filter_value'));
        $sortBy      = $this->request->getGet('sort_by');
        $sortDir     = strtolower((string) $this->request->getGet('sort_dir')) === 'asc' ? 'asc' : 'desc';
        $pageParam   = 'page_aktivitas';
        $page        = (int) ($this->request->getGet($pageParam) ?? 1);

        $allowedFilters = [
            'nama_aktivitas' => 'nama_aktivitas',
            'jenis'          => 'jenis',
            'tanggal'        => 'tanggal',
        ];

        $allowedSorts = [
            'id'             => 'id',
            'nama_aktivitas' => 'nama_aktivitas',
            'tanggal'        => 'tanggal',
            'jam'            => 'jam',
            'jenis'          => 'jenis',
            'created_at'     => 'created_at',
        ];

        if (! array_key_exists($filterField, $allowedFilters)) {
            $filterField = '';
        }

        if (! array_key_exists($sortBy, $allowedSorts)) {
            $sortBy = 'tanggal';
        }

        $query = $this->aktivitasModel;

        if ($search !== '') {
            $query = $query->groupStart()
                ->like('nama_aktivitas', $search)
                ->orLike('deskripsi', $search)
                ->orLike('jenis', $search)
                ->orLike('tanggal', $search)
                ->orLike('jam', $search)
                ->groupEnd();
        }

        if ($filterValue !== '' && $filterField !== '') {
            $query = $query->like($allowedFilters[$filterField], $filterValue);
        }

        $query = $query->orderBy($allowedSorts[$sortBy], $sortDir);

        if ($sortBy === 'tanggal') {
            // Konsistenkan urutan tanggal dengan jam
            $query = $query->orderBy('jam', $sortDir);
        }

        $aktivitas = $query->paginate($perPage, 'aktivitas', $page);

        $aktivitas = array_map(fn ($item) => $this->normalizeAktivitasMedia($item), $aktivitas);

        $data = [
            'title'       => 'Aktivitas - Ridwan',
            'aktivitas'   => $aktivitas,
            'pager'       => $this->aktivitasModel->pager,
            'search'      => $search,
            'filterField' => $filterField,
            'filterValue' => $filterValue,
            'sortBy'      => $sortBy,
            'sortDir'     => $sortDir,
            'perPage'     => $perPage,
            'pageParam'   => $pageParam,
        ];
        return view('aktivitas', $data);
    }

    public function aktivitasDetail(int $id)
    {
        $item = $this->aktivitasModel->find($id);
        if (! $item) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound("Aktivitas dengan ID {$id} tidak ditemukan");
        }

        $item = $this->normalizeAktivitasMedia($item);

        $data = [
            'title'     => ($item['nama_aktivitas'] ?? 'Aktivitas') . ' - Portfolio Digital',
            'aktivitas' => $item,
        ];

        return view('aktivitas_detail', $data);
    }
}
